// $Id: CClass.cpp,v 1.9 2007/02/08 21:06:44 paul Exp $

/*
 * All contents of this source code are copyright 2005 Exp Digital Uk.
 * This source file is covered by the licence conditions of the Infinity API. You should have recieved a copy
 * with the source code. If you didnt, please refer to http://www.expdigital.co.uk
 * All content is the Intellectual property of Exp Digital Uk.
 * Certain sections of this code may come from other sources. They are credited where applicable.
 * If you have comments, suggestions or bug reports please visit http://support.expdigital.co.uk
 */

#include "CClass.hpp"
#include "../IO/CTextStream.hpp"
using Exponent::Basics::CClass;
using Exponent::IO::CTextStream;
#ifndef WIN32
	#include <cxxabi.h>				// Need for name demangling
#endif

//	===========================================================================
CClassManager *CClass::CCLASSCLASS_MANAGER = NULL;

//	===========================================================================
CClass::CClass() 
	  : m_instanceCount(0)
	  , m_maxInstanceCount(0)
{
	strncpy(m_className, "UndefinedClass", CCLASS_MAX_CLASSNAME_LENGTH);
	strncpy(m_parentName, "UndefinedParent", CCLASS_MAX_CLASSNAME_LENGTH);

	if (CCLASSCLASS_MANAGER == NULL)
	{
		CCLASSCLASS_MANAGER = new CClassManager();
	}
}

//	===========================================================================
CClass::CClass(const char *className)
	  : m_instanceCount(0)
	  , m_maxInstanceCount(0)
{
	strncpy(m_className, className, CCLASS_MAX_CLASSNAME_LENGTH);
	strncpy(m_parentName, "UndefinedParent", CCLASS_MAX_CLASSNAME_LENGTH);

	if (CCLASSCLASS_MANAGER == NULL)
	{
		CCLASSCLASS_MANAGER = new CClassManager();
	}
	CCLASSCLASS_MANAGER->registerClass(this);
}

//	===========================================================================
CClass::CClass(const char *className, const char *parentName)
	  : m_instanceCount(0)
	  , m_maxInstanceCount(0)
{
	strncpy(m_className,  className,  CCLASS_MAX_CLASSNAME_LENGTH);
	strncpy(m_parentName, parentName, CCLASS_MAX_CLASSNAME_LENGTH);

	if (CCLASSCLASS_MANAGER == NULL)
	{
		CCLASSCLASS_MANAGER = new CClassManager();
	}
	CCLASSCLASS_MANAGER->registerClass(this);
}

//	===========================================================================
CClass::CClass(const char *templateName, const std::type_info &theType)
	  : m_instanceCount(0)
	  , m_maxInstanceCount(0)
{
	computeTemplateName(templateName, theType, m_className, CCLASS_MAX_CLASSNAME_LENGTH);
	strncpy(m_parentName, "UndefinedParent", CCLASS_MAX_CLASSNAME_LENGTH);

	if (CCLASSCLASS_MANAGER == NULL)
	{
		CCLASSCLASS_MANAGER = new CClassManager();
	}
	CCLASSCLASS_MANAGER->registerClass(this);
}

//	===========================================================================
CClass::CClass(const char *templateName, const std::type_info &theType, const char *parentName)
	  : m_instanceCount(0)
	  , m_maxInstanceCount(0)
{
	computeTemplateName(templateName, theType, m_className, CCLASS_MAX_CLASSNAME_LENGTH);
	strncpy(m_parentName, parentName, CCLASS_MAX_CLASSNAME_LENGTH);

	if (CCLASSCLASS_MANAGER == NULL)
	{
		CCLASSCLASS_MANAGER = new CClassManager();
	}
	CCLASSCLASS_MANAGER->registerClass(this);
}

//	===========================================================================
CClass::CClass(const char *templateName, const std::type_info &theType, const char *parentName, const std::type_info &parentType)
	  : m_instanceCount(0)
	  , m_maxInstanceCount(0)
{
	computeTemplateName(templateName, theType,  m_className,  CCLASS_MAX_CLASSNAME_LENGTH);
	computeTemplateName(parentName, parentType, m_parentName, CCLASS_MAX_CLASSNAME_LENGTH);

	if (CCLASSCLASS_MANAGER == NULL)
	{
		CCLASSCLASS_MANAGER = new CClassManager();
	}
	CCLASSCLASS_MANAGER->registerClass(this);
}

//	===========================================================================
CClass::CClass(const CClass &theClass)
	  : m_instanceCount(0)
	  , m_maxInstanceCount(0)
{
	strncpy(m_className, theClass.m_className, CCLASS_MAX_CLASSNAME_LENGTH);

	if (CCLASSCLASS_MANAGER == NULL)
	{
		CCLASSCLASS_MANAGER = new CClassManager();
	}
	CCLASSCLASS_MANAGER->registerClass(this);
}

//	===========================================================================
CClass::~CClass()
{
	// Notice that we dont delete the class manager, this must be done by the API shutdown routines
}

//	===========================================================================
bool CClass::operator == (const CClass &other) const
{
	return this->isTypeOf(other);
}

//	===========================================================================
CClass &CClass::operator = (const CClass &other)
{
	if (this != &other)
	{
		strncpy(m_className, other.m_className, CCLASS_MAX_CLASSNAME_LENGTH);
	}
	return *this;
}

//	===========================================================================
void CClass::operator ++ (int /* Undefined */)
{
	m_instanceCount++;
	m_maxInstanceCount++;
}

//	===========================================================================
void CClass::operator -- (int /* Undefined */)
{
	m_instanceCount--;
}

//	===========================================================================
void CClass::setClassInformation(const char *className)
{
	if (className)
	{
		strncpy(m_className, className, CCLASS_MAX_CLASSNAME_LENGTH);
	}
}

//	===========================================================================
bool CClass::isTypeOf(const char *className) const
{
	return (strcmp(className, m_className) == 0);
}

//	===========================================================================
bool CClass::isTypeOf(const CClass &other) const
{
	return (strcmp(other.getClassName(), m_className) == 0);
}

//	===========================================================================
bool CClass::isSubClassOf(const CClass &other) const
{
	CClass *parent = CCLASSCLASS_MANAGER->getClassWithName(m_parentName);

	while(parent != NULL)
	{
		if (parent->isTypeOf(other))
		{
			return true;
		}
		else
		{
			parent = CCLASSCLASS_MANAGER->getClassWithName(parent->m_parentName);
		}
	}

	return this->isTypeOf(other);
}

//	===========================================================================
void CClass::deleteClassManager(const char *filePath)
{
	if (CCLASSCLASS_MANAGER && filePath)
	{
		CTextStream stream(filePath, CTextStream::e_output);
		CCLASSCLASS_MANAGER->sortClasses();
		CCLASSCLASS_MANAGER->logClassInformation(&stream);
	}
	FREE_POINTER(CCLASSCLASS_MANAGER);
}

//	===========================================================================
int CClass::compareClasses(const CClass **class1, const CClass **class2)
{
	if (*class1 == *class2)
	{
		return 0;
	}
	else if (*class1 == NULL)
	{
		return 1;
	}
	else if (*class2 == NULL)
	{
		return -1;
	}
	else
	{
		return strcmp(((CClass *)(*class1))->getClassName(), ((CClass *)(*class2))->getClassName());
	}
}

//	===========================================================================
bool CClass::computeTemplateName(const char *templateName, const std::type_info &theType, char *theBuffer, const long bufferSize)
{
	// These are the variables that we will setup
	CString theTemplateName = templateName;
	CString theTypeName;
	CString scopedTypeName;

#ifdef WIN32
	// Store the converted type
	scopedTypeName = theType.name();
#else
	// Try to demangle it
	int status;
	char *buffer = abi::__cxa_demangle(theType.name(), 0, 0, &status);
	
	// If we got a valid buffer back
	if (buffer)
	{
		// Store the demangled version
		scopedTypeName = buffer;
		
		// Delete the buffer, dont like having c,malloced stuff around !yuck!
		free(buffer);
	}
	else
	{
		scopedTypeName = theType.name();
	}
#endif

	// We now have the globally scoped variable.
	// Windows uses this format 'class Namespace::Namespace::ClassName'
	// GCC/XCode uses this format 'Namespace::Namespace::ClassName'

	// First we see if we can find a namespace delimiter
	long typeIndex = scopedTypeName.findBackwards(':');

	// If the index is -1, we didnt find the colon
	if (typeIndex == -1)
	{
		// Then we didnt find a semi colon, maybe its a windows one?
		typeIndex = scopedTypeName.findBackwards(' ');

		// Otherwise we didnt find it, and we fall through to the main string
		if (typeIndex == -1)
		{
			theTypeName = scopedTypeName;
		}
		else
		{
			theTypeName = scopedTypeName.getSubString(typeIndex + 1, scopedTypeName.getNumberOfCharacters() - 1);
		}

	}
	else
	{
		theTypeName = scopedTypeName.getSubString(typeIndex + 1, scopedTypeName.getNumberOfCharacters() - 1);
	}

	// At this point we have our nicely seperated string type
	// Next thing is to work out what the template part is

	// Find the opening template
	const long typeNameTemplateStart = theTemplateName.findForward('<');

	// If we failed to find, then its not a template
	if (typeNameTemplateStart == -1)
	{
		CString temp = templateName;
		temp.getString(theBuffer, bufferSize);
		return true;
		//return CClass(templateName);
	}
	
	// Other wise we found an index and we want to select it
	CString myTemplateName = theTemplateName.getSubString(0, typeNameTemplateStart - 1);
	myTemplateName.appendString("<");
	myTemplateName.appendString(theTypeName);
	myTemplateName.appendString(">");
	myTemplateName.getString(theBuffer, bufferSize);
	return true;
	//return CClass(myTemplateName.getString());
}